DROP PROCEDURE [hips].[HospitalPatientMove]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Simon Biber
-- Create date:   30 May 2013
-- Description:   Moves all the MRNs for a hospital patient, from one enterprise ID to another.
--                This includes the current MRN as well as any previously merged ones.
--                If as a result there are no MRNs from any hospital on the old enterprise ID,
--                then moves the IHI to the history as it has been fully resolved.
-- Modified:
-- Simon Biber - 19 June 2013 - Create merge conflict alert if merging patients
--                              who have two different IHI numbers.
-- =============================================
CREATE PROCEDURE [hips].[HospitalPatientMove]
(
	 @PatientId INT
	,@HospitalId INT
    ,@FromPatientMasterId INT
	,@ToPatientMasterId INT
	,@UserModified varchar(256)
	,@DateModified DATETIME
)
AS
BEGIN
    SET NOCOUNT ON
    DECLARE @intErrorCode INT, @FromIhi VARCHAR(16), @ToIhi VARCHAR(16), @ConflictStatus INT, @FromMrn VARCHAR(20), @ToMrn VARCHAR(20), @Message VARCHAR(512), @HospitalDesc VARCHAR(60)

	SET @ConflictStatus = 102
	SET @FromIhi = (SELECT Ihi FROM hips.PatientMasterIhi WHERE PatientMasterId = @FromPatientMasterId)
	SET @ToIhi = (SELECT Ihi FROM hips.PatientMasterIhi WHERE PatientMasterId = @ToPatientMasterId)
	SET @FromMrn = (SELECT TOP 1 Mrn FROM hips.HospitalPatient WHERE HospitalId = @HospitalId AND PatientMasterId = @FromPatientMasterId ORDER BY DateModified DESC)
	SET @ToMrn = (SELECT TOP 1 Mrn FROM hips.HospitalPatient WHERE HospitalId = @HospitalId AND PatientMasterId = @ToPatientMasterId ORDER BY DateModified DESC)
	SET @HospitalDesc = (SELECT [Description] FROM hips.Hospital WHERE HospitalId = @HospitalId)
		
    SET @intErrorCode = 0
    IF NOT EXISTS (SELECT TOP 1 [DateModified]
                FROM [hips].[HospitalPatient]
        WHERE [PatientId] = @PatientId
        AND [DateModified] = @DateModified)
    BEGIN
        RAISERROR 50001 'Record has already been updated!'
        SET @intErrorCode = @@ERROR
    END
    IF @intErrorCode = 0
    BEGIN
		IF (@FromIhi <> @ToIhi) AND (@ToMrn IS NOT NULL)
		BEGIN
				-- We are moving to a PM that already has an MRN from this hospital.
				-- Create IHI Conflict alert against both patient masters
				UPDATE	[hips].[PatientMasterIhi]
				SET		[IhiStatusId] = @ConflictStatus
						,[DateModified] = GETDATE()
						,[UserModified] = @UserModified
				WHERE [PatientMasterId]	= @FromPatientMasterId
				OR	[PatientMasterId] = @ToPatientMasterId
						
				SET @Message = 'IHI conflict merging MRN '
					+ @FromMrn + ' at ' + @HospitalDesc + ' with IHI ' + @FromIhi + ' into MRN '
					+ @ToMrn + ' at ' + @HospitalDesc + ' with IHI ' + @ToIhi
				
				INSERT INTO [hips].IhiLookupAlert
						 ( PatientMasterId ,
						   Message ,
						   Request ,
						   Response ,
						   DateModified ,
						   UserModified ,
						   DateCreated ,
						   UserCreated
						 )
				 VALUES  ( @FromPatientMasterId , -- PatientMasterId - int
						   @Message, -- Message - varchar(512)
						   NULL, -- Request - xml
						   NULL, -- Response - xml
						   GETDATE() , -- DateModified - datetime
						   @UserModified , -- UserModified - varchar(256)
						   GETDATE() , -- DateCreated - datetime
						   @UserModified  -- UserCreated - varchar(256)
						 )
				INSERT INTO [hips].IhiLookupAlert
						 ( PatientMasterId ,
						   Message ,
						   Request ,
						   Response ,
						   DateModified ,
						   UserModified ,
						   DateCreated ,
						   UserCreated
						 )
				 VALUES  ( @ToPatientMasterId , -- PatientMasterId - int
						   @Message, -- Message - varchar(512)
						   NULL, -- Request - xml
						   NULL, -- Response - xml
						   GETDATE() , -- DateModified - datetime
						   @UserModified , -- UserModified - varchar(256)
						   GETDATE() , -- DateCreated - datetime
						   @UserModified  -- UserCreated - varchar(256)
						 )
		END

		UPDATE	hips.HospitalPatient
		SET		PatientMasterId = @ToPatientMasterId,
				DateModified = @DateModified,
				UserModified = @UserModified
		WHERE	PatientMasterId = @FromPatientMasterId
		AND		HospitalId = @HospitalId
   
	    SELECT @intErrorCode = @@ERROR

		IF @intErrorCode = 0
		BEGIN
			IF NOT EXISTS (  -- No MRNs left on the source PM
				SELECT	PatientId
				FROM	hips.HospitalPatient
				WHERE	PatientMasterId = @FromPatientMasterId)
			AND NOT EXISTS ( -- and the source PM's IHI is not in conflict
				SELECT	IhiStatusId
				FROM	hips.PatientMasterIhi
				WHERE	PatientMasterId = @FromPatientMasterId
				AND		IhiStatusId = @ConflictStatus)
			BEGIN
				INSERT	hips.PatientMasterIhiHistory				
				SELECT	 PatientMasterId
						,Ihi
						,IhiRecordStatusId
						,IhiStatusId
						,GETDATE()
						,@UserModified
						,GETDATE()
						,@UserModified
				FROM	hips.PatientMasterIhi
				WHERE	PatientMasterId = @FromPatientMasterId

				SELECT @intErrorCode = @@ERROR

				IF @intErrorCode = 0
				BEGIN
					DELETE	hips.PatientMasterIhi
					WHERE	PatientMasterId = @FromPatientMasterId

					SELECT @intErrorCode = @@ERROR
				END
			END
		END
    END

	IF @intErrorCode = 0
    BEGIN
        SELECT 
           PatientId,
           PatientMasterId,
           HospitalId,
           Mrn,
           DateCreated,
           UserCreated,
           DateModified,
           UserModified
         FROM [hips].[HospitalPatient]
         WHERE [PatientId] = @PatientId
    END

    RETURN(@intErrorCode)
END
GO

